/*  Icecast-Plugin
 *  (C) copyright 2008 based of FileWriter-plugin
 *
 *  Original Out-Lame-Plugin:
 *  (C) copyright 2002 Lars Siebold <khandha5@gmx.net>
 *  (C) copyright 2006-2007 porting to audacious by Yoshiki Yazawa <yaz@cc.rim.or.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "../filewriter/filewriter.h"
#include "../filewriter/plugins.h"
#include <shout/shout.h>

struct format_info input;

static GtkWidget *configure_win = NULL, *configure_vbox;
static GtkWidget *addr_hbox, *addr_label, *addr_entry, *port_spin;
static GtkWidget *configure_bbox, *configure_ok, *configure_cancel;
static guint ice_tid=0;

static gint ice_close_timeout;

static GtkWidget *streamformat_hbox, *streamformat_label, *streamformat_combo, *plugin_button;

enum streamformat_t
{
#ifdef FILEWRITER_MP3
    MP3,
#endif
#ifdef FILEWRITER_VORBIS
    VORBIS,
#endif
    streamformat_MAX
};

static gint streamformat = VORBIS;

static unsigned int streamformat_shout[] =
{
#ifdef FILEWRITER_MP3
    SHOUT_FORMAT_MP3,
#endif
#ifdef FILEWRITER_VORBIS
    SHOUT_FORMAT_OGG
#endif
};

static FileWriter plugin;
static uint8_t outputbuffer[4096];
static guint outputlength=0;
static gchar *server_address = NULL;
static gint server_port=8000;

VFSFile *output_file = NULL;
guint64 written = 0;
guint64 offset = 0;
Tuple *tuple = NULL;
static shout_t *shout = NULL;

static void ice_init(void);
static void ice_cleanup(void);
static void ice_about(void);
static gint ice_open(AFormat fmt, gint rate, gint nch);
static void ice_write(void *ptr, gint length);
static gint ice_write_output(void *ptr, gint length);
static void ice_close(void);
static void ice_flush(gint time);
static void ice_pause(short p);
static gint ice_free(void);
static gint ice_playing(void);
static gint ice_get_written_time(void);
static gint ice_get_output_time(void);
static void ice_configure(void);
/*static int ice_mod_samples(gpointer * d, gint length, AFormat afmt, gint srate, gint nch);*/

OutputPlugin ice_op =
{
    .description = "Icecast Plugin",
    .init = ice_init,
    .cleanup = ice_cleanup,
    .about = ice_about,
    .configure = ice_configure,
    .open_audio = ice_open,
    .write_audio = ice_write,
    .close_audio = ice_close,
    .flush = ice_flush,
    .pause = ice_pause,
    .buffer_free = ice_free,
    .buffer_playing = ice_playing,
    .output_time = ice_get_output_time,
    .written_time = ice_get_written_time
};
/*
EffectPlugin ice_ep =
{
        .description = "Icecast Plugin",
        .init = ice_init,
        .cleanup = ice_cleanup,
        .about = ice_about,
        .configure = ice_configure,
        .mod_samples = ice_mod_samples,
};
*/
OutputPlugin *ice_oplist[] = { &ice_op, NULL };

SIMPLE_OUTPUT_PLUGIN(icecast, ice_oplist);

static void set_plugin(void)
{
    if (streamformat < 0 || streamformat >= streamformat_MAX)
        streamformat = 0;

#ifdef FILEWRITER_MP3
    if (streamformat == MP3)
        plugin = mp3_plugin;
#endif
#ifdef FILEWRITER_VORBIS
    if (streamformat == VORBIS)
        plugin = vorbis_plugin;
#endif
}

static void ice_init(void)
{
    ConfigDb *db;
    puts("ICE_INIT");
    shout_init();
    printf("Using libshout %s\n", shout_version(NULL, NULL, NULL));

    db = aud_cfg_db_open();
    aud_cfg_db_get_int(db, "icecast", "streamformat", &streamformat);
    aud_cfg_db_get_string(db, "icecast", "server_address", &server_address);
    aud_cfg_db_get_int(db, "icecast", "server_port", &server_port);
    if (!server_port) server_port=8000;
    aud_cfg_db_get_int(db, "icecast", "timeout", &ice_close_timeout);
    if (!ice_close_timeout) ice_close_timeout=5;
    aud_cfg_db_close(db);

    set_plugin();
    if (plugin.init)
        plugin.init(&ice_write_output);
}

static void ice_cleanup(void)
{
    if (shout)
    {
        shout_close(shout);
    }
    shout_shutdown();
}

void ice_about(void)
{
    static GtkWidget *dialog;

    if (dialog != NULL)
        return;

    dialog = audacious_info_dialog(_("About Icecast-Plugin"),
                               _("Icecast-Plugin\n\n"
                               "This program is free software; you can redistribute it and/or modify\n"
                               "it under the terms of the GNU General Public License as published by\n"
                               "the Free Software Foundation; either version 2 of the License, or\n"
                               "(at your option) any later version.\n"
                               "\n"
                               "This program is distributed in the hope that it will be useful,\n"
                               "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
                               "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
                               "GNU General Public License for more details.\n"
                               "\n"
                               "You should have received a copy of the GNU General Public License\n"
                               "along with this program; if not, write to the Free Software\n"
                               "Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,\n"
                               "USA."), _("Ok"), FALSE, NULL, NULL);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(gtk_widget_destroyed), &dialog);
}

static gint ice_open(AFormat fmt, gint rate, gint nch)
{
    gint rv;
    gint pos;
    Playlist *playlist;

    if (ice_tid)
    {
	g_source_remove(ice_tid);
	ice_tid = 0;
    }

    input.format = fmt;
    input.frequency = rate;
    input.channels = nch;

    playlist = aud_playlist_get_active();
    if(!playlist)
        return 0;

    pos = aud_playlist_get_position(playlist);
    tuple = aud_playlist_get_tuple(playlist, pos);

    if (!shout)
    {
        rv = (plugin.open)();

        if (!(shout = shout_new()))
            return 0;

        if (shout_set_host(shout, server_address) != SHOUTERR_SUCCESS)
        {
            printf("Error setting hostname: %s\n", shout_get_error(shout));
            return 0;
        }

        if (shout_set_protocol(shout, SHOUT_PROTOCOL_HTTP) != SHOUTERR_SUCCESS)
        {
            printf("Error setting protocol: %s\n", shout_get_error(shout));
            return 0;
        }

        if (shout_set_port(shout, server_port) != SHOUTERR_SUCCESS)
        {
            printf("Error setting port: %s\n", shout_get_error(shout));
            return 0;
        }

        if (shout_set_password(shout, "password") != SHOUTERR_SUCCESS)
        {
            printf("Error setting password: %s\n", shout_get_error(shout));
            return 0;
        }

        if (shout_set_mount(shout, "/test") != SHOUTERR_SUCCESS)
        {
            printf("Error setting mount: %s\n", shout_get_error(shout));
            return 0;
        }

        if (shout_set_user(shout, "source") != SHOUTERR_SUCCESS)
        {
            printf("Error setting user: %s\n", shout_get_error(shout));
            return 0;
        }

        if (shout_set_format(shout, streamformat_shout[streamformat]) != SHOUTERR_SUCCESS)
        {
            printf("Error setting user: %s\n", shout_get_error(shout));
            return 0;
        }

        if (shout_open(shout) != SHOUTERR_SUCCESS)
        {
            printf("Error connecting to server: %s\n", shout_get_error(shout));
            return 0;
        }
    }
    else
        rv = 1;

    {
        shout_metadata_t *sm = NULL;
        sm = shout_metadata_new();
        if (sm)
        {
            shout_metadata_add(sm, "charset", "UTF-8");
            shout_metadata_add(sm, "title", aud_tuple_get_string(tuple, FIELD_TITLE, NULL));
            shout_metadata_add(sm, "artist", aud_tuple_get_string(tuple, FIELD_ARTIST, NULL));
            shout_set_metadata(shout, sm);
            shout_metadata_free(sm);
        }
    }

    puts("ICE_OPEN");
    return rv;
}

static void convert_buffer(gpointer buffer, gint length)
{
    gint i;

    if (input.format == FMT_S8)
    {
        guint8 *ptr1 = buffer;
        gint8 *ptr2 = buffer;

        for (i = 0; i < length; i++)
            *(ptr1++) = *(ptr2++) ^ 128;
    }
    if (input.format == FMT_S16_BE)
    {
        gint16 *ptr = buffer;

        for (i = 0; i < length >> 1; i++, ptr++)
            *ptr = GUINT16_SWAP_LE_BE(*ptr);
    }
    if (input.format == FMT_S16_NE)
    {
        gint16 *ptr = buffer;

        for (i = 0; i < length >> 1; i++, ptr++)
            *ptr = GINT16_TO_LE(*ptr);
    }
    if (input.format == FMT_U16_BE)
    {
        gint16 *ptr1 = buffer;
        guint16 *ptr2 = buffer;

        for (i = 0; i < length >> 1; i++, ptr2++)
            *(ptr1++) = GINT16_TO_LE(GUINT16_FROM_BE(*ptr2) ^ 32768);
    }
    if (input.format == FMT_U16_LE)
    {
        gint16 *ptr1 = buffer;
        guint16 *ptr2 = buffer;

        for (i = 0; i < length >> 1; i++, ptr2++)
            *(ptr1++) = GINT16_TO_LE(GUINT16_FROM_LE(*ptr2) ^ 32768);
    }
    if (input.format == FMT_U16_NE)
    {
        gint16 *ptr1 = buffer;
        guint16 *ptr2 = buffer;

        for (i = 0; i < length >> 1; i++, ptr2++)
            *(ptr1++) = GINT16_TO_LE((*ptr2) ^ 32768);
    }
}

static void ice_write(void *ptr, gint length)
{
    if (input.format == FMT_S8 || input.format == FMT_S16_BE ||
        input.format == FMT_U16_LE || input.format == FMT_U16_BE ||
        input.format == FMT_U16_NE)
        convert_buffer(ptr, length);
#ifdef WORDS_BIGENDIAN
    if (input.format == FMT_S16_NE)
        convert_buffer(ptr, length);
#endif

    plugin.write(ptr, length);
}

static gint ice_real_write(void* ptr, gint length)
{
    int ret;
    if (!length) return length;
    ret = shout_send(shout, ptr, length);
    //shout_send_raw(shout, ptr, length);
    shout_sync(shout);
    printf("ice_write[%d:%d](", ret, length);
    {
        int i;
        for (i=0;(i<length)&&(i<16);i++)   printf("%c",g_ascii_isprint(((char*)ptr)[i])?(((char*)ptr)[i]):'.');
    }
    printf(")\n");
    return 0;
    //return ret;
}

static gint ice_write_output(void *ptr, gint length)
{
    if ((!shout) || (!length)) return 0;
    printf("outputlength=%d, length=%d...",outputlength, length);
    if ((outputlength>4000)||((outputlength+length)>4096))
    {
        printf("flushing\n");
        outputlength=ice_real_write(outputbuffer, outputlength);
    }
    {
        if (length>4096)
        {
            printf("data too long, flushing\n");
            ice_real_write(ptr, length);
        }
        else
        {
            printf("adding\n");
            memcpy(&(outputbuffer[outputlength]), ptr, length);
            outputlength+=length;
        }
    }
    return length;
}

static gboolean ice_real_close(gpointer data)
{
    plugin.close();

    if (shout)
    {
        written = 0;
        shout_close(shout);
    }
    shout = NULL;
    ice_tid=0;
    puts("ICE_REAL_CLOSE");
    return FALSE;
}


static void ice_close(void)
{
    if (ice_tid) g_source_remove(ice_tid);
    ice_tid=g_timeout_add_seconds(3, ice_real_close, NULL);
    puts("ICE_CLOSE: starting timer");
}

static void ice_flush(gint time)
{
    if (time < 0)
        return;

    plugin.flush();
    ice_open(input.format, input.frequency, input.channels);

    offset = time;
}

static void ice_pause(short p)
{
}

static gint ice_free(void)
{
    return plugin.free();
}

static gint ice_playing(void)
{
    return plugin.playing();
}

static gint ice_get_written_time(void)
{
    return plugin.get_written_time();
}

static gint ice_get_output_time(void)
{
    return ice_get_written_time();
}

static void configure_ok_cb(gpointer data)
{
    ConfigDb *db;

    streamformat = gtk_combo_box_get_active(GTK_COMBO_BOX(streamformat_combo));

    g_free(server_address);
    server_address = g_strdup(gtk_entry_get_text(GTK_ENTRY(addr_entry)));

    server_port = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(port_spin));

    db = aud_cfg_db_open();
    aud_cfg_db_set_int(db, "icecast", "streamformat", streamformat);
    aud_cfg_db_set_string(db, "icecast", "server_address", server_address);
    aud_cfg_db_set_int(db, "icecast", "server_port", server_port);
    aud_cfg_db_set_int(db, "icecast", "timeout", ice_close_timeout);

    aud_cfg_db_close(db);

    gtk_widget_destroy(configure_win);
}

static void streamformat_cb(GtkWidget *combo, gpointer data)
{
    streamformat = gtk_combo_box_get_active(GTK_COMBO_BOX(streamformat_combo));
    set_plugin();
    if (plugin.init)
        plugin.init(&ice_write_output);

    gtk_widget_set_sensitive(plugin_button, plugin.configure != NULL);
}

static void plugin_configure_cb(GtkWidget *button, gpointer data)
{
    if (plugin.configure)
        plugin.configure();
}

static void configure_destroy(void)
{
}

static void ice_configure(void)
{
    if (!configure_win)
    {
        configure_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);

        gtk_signal_connect(GTK_OBJECT(configure_win), "destroy",
                           GTK_SIGNAL_FUNC(configure_destroy), NULL);
        gtk_signal_connect(GTK_OBJECT(configure_win), "destroy",
                           GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                           &configure_win);

        gtk_window_set_title(GTK_WINDOW(configure_win),
                             _("Icecast Configuration"));
        gtk_window_set_position(GTK_WINDOW(configure_win), GTK_WIN_POS_MOUSE);

        gtk_container_set_border_width(GTK_CONTAINER(configure_win), 10);

        configure_vbox = gtk_vbox_new(FALSE, 10);
        gtk_container_add(GTK_CONTAINER(configure_win), configure_vbox);


        streamformat_hbox = gtk_hbox_new(FALSE, 5);
        gtk_box_pack_start(GTK_BOX(configure_vbox), streamformat_hbox, FALSE, FALSE, 0);

        streamformat_label = gtk_label_new(_("Output stream format:"));
        gtk_box_pack_start(GTK_BOX(streamformat_hbox), streamformat_label, FALSE, FALSE, 0);

        streamformat_combo = gtk_combo_box_new_text();
#ifdef FILEWRITER_MP3
        gtk_combo_box_append_text(GTK_COMBO_BOX(streamformat_combo), "MP3");
#endif
#ifdef FILEWRITER_VORBIS
        gtk_combo_box_append_text(GTK_COMBO_BOX(streamformat_combo), "Vorbis");
#endif
        gtk_box_pack_start(GTK_BOX(streamformat_hbox), streamformat_combo, FALSE, FALSE, 0);
        gtk_combo_box_set_active(GTK_COMBO_BOX(streamformat_combo), streamformat);
        g_signal_connect(G_OBJECT(streamformat_combo), "changed", G_CALLBACK(streamformat_cb), NULL);

        plugin_button = gtk_button_new_with_label(_("Configure"));
        gtk_widget_set_sensitive(plugin_button, plugin.configure != NULL);
        g_signal_connect(G_OBJECT(plugin_button), "clicked", G_CALLBACK(plugin_configure_cb), NULL);
        gtk_box_pack_end(GTK_BOX(streamformat_hbox), plugin_button, FALSE, FALSE, 0);




        gtk_box_pack_start(GTK_BOX(configure_vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

        addr_hbox = gtk_hbox_new(FALSE, 5);
        gtk_box_pack_start(GTK_BOX(configure_vbox), addr_hbox, FALSE, FALSE, 0);

        addr_label = gtk_label_new(_("Server address:"));
        gtk_box_pack_start(GTK_BOX(addr_hbox), addr_label, FALSE, FALSE, 0);

        addr_entry = gtk_entry_new();

	gtk_entry_set_text(GTK_ENTRY(addr_entry), server_address);

        gtk_box_pack_start(GTK_BOX(addr_hbox), addr_entry, TRUE, TRUE, 0);

        port_spin = gtk_spin_button_new_with_range(0.0, 65535.0, 1.0);

        gtk_spin_button_set_digits(GTK_SPIN_BUTTON(port_spin), 0);

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(port_spin), (gdouble)server_port);

        gtk_box_pack_start(GTK_BOX(addr_hbox), port_spin, TRUE, TRUE, 0);

        gtk_box_pack_start(GTK_BOX(configure_vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

        configure_bbox = gtk_hbutton_box_new();
        gtk_button_box_set_layout(GTK_BUTTON_BOX(configure_bbox),
                                  GTK_BUTTONBOX_END);
        gtk_button_box_set_spacing(GTK_BUTTON_BOX(configure_bbox), 5);
        gtk_box_pack_start(GTK_BOX(configure_vbox), configure_bbox,
                           FALSE, FALSE, 0);

        configure_cancel = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
        gtk_signal_connect_object(GTK_OBJECT(configure_cancel), "clicked",
                                  GTK_SIGNAL_FUNC(gtk_widget_destroy),
                                  GTK_OBJECT(configure_win));
        gtk_box_pack_start(GTK_BOX(configure_bbox), configure_cancel,
                           TRUE, TRUE, 0);

        configure_ok = gtk_button_new_from_stock(GTK_STOCK_OK);
        gtk_signal_connect(GTK_OBJECT(configure_ok), "clicked",
                           GTK_SIGNAL_FUNC(configure_ok_cb), NULL);
        gtk_box_pack_start(GTK_BOX(configure_bbox), configure_ok,
                           TRUE, TRUE, 0);

        gtk_widget_show_all(configure_win);
    }
}
