/* GnuTLS glue for GNU Emacs.
   Copyright (C) 2010  Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs.  If not, see <http://www.gnu.org/licenses/>.  */

#include <config.h>
#include <errno.h>
#include <setjmp.h>

#include "lisp.h"
#include "process.h"

#ifdef HAVE_GNUTLS
#include <gnutls/gnutls.h>

Lisp_Object Qgnutls_code;
Lisp_Object Qgnutls_anon, Qgnutls_x509pki;
Lisp_Object Qgnutls_e_interrupted, Qgnutls_e_again,
  Qgnutls_e_invalid_session, Qgnutls_e_not_ready_for_handshake;
int global_initialized;

void
emacs_gnutls_handshake (struct Lisp_Process *proc)
{
  gnutls_session_t state = proc->gnutls_state;
  int ret;

  if (proc->gnutls_initstage < GNUTLS_STAGE_HANDSHAKE_CANDO)
    return;

  if (proc->gnutls_initstage < GNUTLS_STAGE_TRANSPORT_POINTERS_SET)
    {
      gnutls_transport_set_ptr2 (state,
				 (gnutls_transport_ptr_t) (long) proc->infd,
				 (gnutls_transport_ptr_t) (long) proc->outfd);

      proc->gnutls_initstage = GNUTLS_STAGE_TRANSPORT_POINTERS_SET;
    }

  ret = gnutls_handshake (state);
  proc->gnutls_initstage = GNUTLS_STAGE_HANDSHAKE_TRIED;

  if (ret == GNUTLS_E_SUCCESS)
    {
      /* here we're finally done.  */
      proc->gnutls_initstage = GNUTLS_STAGE_READY;
    }
}

int
emacs_gnutls_write (int fildes, struct Lisp_Process *proc, char *buf,
                    unsigned int nbyte)
{
  register int rtnval, bytes_written;
  gnutls_session_t state = proc->gnutls_state;

  if (proc->gnutls_initstage != GNUTLS_STAGE_READY)
    return -1;

  bytes_written = 0;

  while (nbyte > 0)
    {
      rtnval = gnutls_write (state, buf, nbyte);

      if (rtnval == -1)
        {
          if (errno == EINTR)
            continue;
          else
            return (bytes_written ? bytes_written : -1);
        }

      buf += rtnval;
      nbyte -= rtnval;
      bytes_written += rtnval;
    }
  fsync (STDOUT_FILENO);

  return (bytes_written);
}

int
emacs_gnutls_read (int fildes, struct Lisp_Process *proc, char *buf,
                   unsigned int nbyte)
{
  register int rtnval;
  gnutls_session_t state = proc->gnutls_state;

  if (proc->gnutls_initstage != GNUTLS_STAGE_READY)
    {
      emacs_gnutls_handshake (proc);
      return -1;
    }

  rtnval = gnutls_read (state, buf, nbyte);
  if (rtnval >= 0)
    return rtnval;
  else
    return 0;
}

/* convert an integer error to a Lisp_Object; it will be either a
   known symbol like `gnutls_e_interrupted' and `gnutls_e_again' or
   simply the integer value of the error.  GNUTLS_E_SUCCESS is mapped
   to Qt.  */
Lisp_Object gnutls_make_error (int error)
{
  switch (error)
    {
    case GNUTLS_E_SUCCESS:
      return Qt;
    case GNUTLS_E_AGAIN:
      return Qgnutls_e_again;
    case GNUTLS_E_INTERRUPTED:
      return Qgnutls_e_interrupted;
    case GNUTLS_E_INVALID_SESSION:
      return Qgnutls_e_invalid_session;
    }

  return make_number (error);
}

DEFUN ("gnutls-get-initstage", Fgnutls_get_initstage, Sgnutls_get_initstage, 1, 1, 0,
       doc: /* Return the GnuTLS init stage of PROCESS.
See also `gnutls-boot'.  */)
    (Lisp_Object proc)
{
  CHECK_PROCESS (proc);

  return make_number (GNUTLS_INITSTAGE (proc));
}

DEFUN ("gnutls-errorp", Fgnutls_errorp, Sgnutls_errorp, 1, 1, 0,
       doc: /* Returns t if ERROR (as generated by gnutls_make_error)
indicates a GnuTLS problem.  */)
    (Lisp_Object error)
{
  if (EQ (error, Qt)) return Qnil;

  return Qt;
}

DEFUN ("gnutls-error-fatalp", Fgnutls_error_fatalp, Sgnutls_error_fatalp, 1, 1, 0,
       doc: /* Checks if ERROR is fatal.
ERROR is an integer or a symbol with an integer `gnutls-code' property.  */)
    (Lisp_Object err)
{
  Lisp_Object code;

  if (EQ (err, Qt)) return Qnil;

  if (SYMBOLP (err))
    {
      code = Fget (err, Qgnutls_code);
      if (NUMBERP (code))
	{
	  err = code;
	}
      else
	{
	  error ("Symbol has no numeric gnutls-code property");
	}
    }

  if (!NUMBERP (err))
    error ("Not an error symbol or code");

  if (0 == gnutls_error_is_fatal (XINT (err)))
    return Qnil;

  return Qt;
}

DEFUN ("gnutls-error-string", Fgnutls_error_string, Sgnutls_error_string, 1, 1, 0,
       doc: /* Returns a description of ERROR.
ERROR is an integer or a symbol with an integer `gnutls-code' property.  */)
    (Lisp_Object err)
{
  Lisp_Object code;

  if (EQ (err, Qt)) return build_string ("Not an error");

  if (SYMBOLP (err))
    {
      code = Fget (err, Qgnutls_code);
      if (NUMBERP (code))
	{
	  err = code;
	}
      else
	{
	  return build_string ("Symbol has no numeric gnutls-code property");
	}
    }

  if (!NUMBERP (err))
    return build_string ("Not an error symbol or code");

  return build_string (gnutls_strerror (XINT (err)));
}

DEFUN ("gnutls-deinit", Fgnutls_deinit, Sgnutls_deinit, 1, 1, 0,
       doc: /* Deallocate GNU TLS resources associated with PROCESS.
See also `gnutls-init'.  */)
    (Lisp_Object proc)
{
  gnutls_session_t state;

  CHECK_PROCESS (proc);
  state = XPROCESS (proc)->gnutls_state;

  if (GNUTLS_INITSTAGE (proc) >= GNUTLS_STAGE_INIT)
    {
      gnutls_deinit (state);
      GNUTLS_INITSTAGE (proc) = GNUTLS_STAGE_INIT - 1;
    }

  return Qt;
}

/* Initializes global GNU TLS state to defaults.
Call `gnutls-global-deinit' when GNU TLS usage is no longer needed.
Returns zero on success.  */
Lisp_Object gnutls_emacs_global_init (void)
{
  int ret = GNUTLS_E_SUCCESS;

  if (!global_initialized)
    ret = gnutls_global_init ();

  global_initialized = 1;

  return gnutls_make_error (ret);
}

/* Deinitializes global GNU TLS state.
See also `gnutls-global-init'.  */
Lisp_Object gnutls_emacs_global_deinit (void)
{
  if (global_initialized)
    gnutls_global_deinit ();

  global_initialized = 0;

  return gnutls_make_error (GNUTLS_E_SUCCESS);
}

static void gnutls_log_function (int level, const char* string)
{
  message("gnutls.c: [%d] %s", level, string);
}

DEFUN ("gnutls-boot", Fgnutls_boot, Sgnutls_boot, 3, 7, 0,
       doc: /* Initializes client-mode GnuTLS for process PROC.
Currently only client mode is supported.  Returns a success/failure
value you can check with `gnutls-errorp'.

PRIORITY_STRING is a string describing the priority.
TYPE is either `gnutls-anon' or `gnutls-x509pki'.
TRUSTFILE is a PEM encoded trust file for `gnutls-x509pki'.
KEYFILE is ... for `gnutls-x509pki' (TODO).
CALLBACK is ... for `gnutls-x509pki' (TODO).
LOGLEVEL is the debug level requested from GnuTLS, try 4.

LOGLEVEL will be set for this process AND globally for GnuTLS.  So if
you set it higher or lower at any point, it affects global debugging.

Note that the priority is set on the client.  The server does not use
the protocols's priority except for disabling protocols that were not
specified.

Processes must be initialized with this function before other GNU TLS
functions are used.  This function allocates resources which can only
be deallocated by calling `gnutls-deinit' or by calling it again.

Each authentication type may need additional information in order to
work.  For X.509 PKI (`gnutls-x509pki'), you need TRUSTFILE and
KEYFILE and optionally CALLBACK.  */)
    (Lisp_Object proc, Lisp_Object priority_string, Lisp_Object type,
     Lisp_Object trustfile, Lisp_Object keyfile, Lisp_Object callback,
     Lisp_Object loglevel)
{
  int ret = GNUTLS_E_SUCCESS;

  int max_log_level = 0;

  /* TODO: GNUTLS_X509_FMT_DER is also an option.  */
  int file_format = GNUTLS_X509_FMT_PEM;

  gnutls_session_t state;
  gnutls_certificate_credentials_t x509_cred;
  gnutls_anon_client_credentials_t anon_cred;
  Lisp_Object global_init;

  CHECK_PROCESS (proc);
  CHECK_SYMBOL (type);
  CHECK_STRING (priority_string);

  state = XPROCESS (proc)->gnutls_state;
  XPROCESS (proc)->gnutls_p = 1;

  if (NUMBERP (loglevel))
    {
      gnutls_global_set_log_function (gnutls_log_function);
      gnutls_global_set_log_level (XINT (loglevel));
      max_log_level = XINT (loglevel);
      XPROCESS (proc)->gnutls_log_level = max_log_level;
    }

  /* always initialize globals.  */
  global_init = gnutls_emacs_global_init ();
  if (! NILP (Fgnutls_errorp (global_init)))
    return global_init;

  /* deinit and free resources.  */
  if (GNUTLS_INITSTAGE (proc) >= GNUTLS_STAGE_CRED_ALLOC)
    {
      GNUTLS_LOG (1, max_log_level, "deallocating credentials");

      if (EQ (type, Qgnutls_x509pki))
	{
          GNUTLS_LOG (2, max_log_level, "deallocating x509 credentials");
          x509_cred = XPROCESS (proc)->gnutls_x509_cred;
          gnutls_certificate_free_credentials (x509_cred);
	}
      else if (EQ (type, Qgnutls_anon))
	{
          GNUTLS_LOG (2, max_log_level, "deallocating anon credentials");
          anon_cred = XPROCESS (proc)->gnutls_anon_cred;
          gnutls_anon_free_client_credentials (anon_cred);
	}
      else
	{
          error ("unknown credential type");
          ret = GNUTLS_EMACS_ERROR_INVALID_TYPE;
	}

      if (GNUTLS_INITSTAGE (proc) >= GNUTLS_STAGE_INIT)
	{
          GNUTLS_LOG (1, max_log_level, "deallocating x509 credentials");
          Fgnutls_deinit (proc);
	}
    }

  GNUTLS_INITSTAGE (proc) = GNUTLS_STAGE_EMPTY;

  GNUTLS_LOG (1, max_log_level, "allocating credentials");

  if (EQ (type, Qgnutls_x509pki))
    {
      GNUTLS_LOG (2, max_log_level, "allocating x509 credentials");
      x509_cred = XPROCESS (proc)->gnutls_x509_cred;
      if (gnutls_certificate_allocate_credentials (&x509_cred) < 0)
        memory_full ();
    }
  else if (EQ (type, Qgnutls_anon))
    {
      GNUTLS_LOG (2, max_log_level, "allocating anon credentials");
      anon_cred = XPROCESS (proc)->gnutls_anon_cred;
      if (gnutls_anon_allocate_client_credentials (&anon_cred) < 0)
        memory_full ();
    }
  else
    {
      error ("unknown credential type");
      ret = GNUTLS_EMACS_ERROR_INVALID_TYPE;
    }

  if (ret < GNUTLS_E_SUCCESS)
    return gnutls_make_error (ret);

  GNUTLS_INITSTAGE (proc) = GNUTLS_STAGE_CRED_ALLOC;

  if (EQ (type, Qgnutls_x509pki))
    {
      if (STRINGP (trustfile))
	{
          GNUTLS_LOG (1, max_log_level, "setting the trustfile");
          ret = gnutls_certificate_set_x509_trust_file
            (x509_cred,
             SDATA (trustfile),
             file_format);

          if (ret < GNUTLS_E_SUCCESS)
            return gnutls_make_error (ret);
	}

      if (STRINGP (keyfile))
	{
          GNUTLS_LOG (1, max_log_level, "setting the keyfile");
          ret = gnutls_certificate_set_x509_crl_file
            (x509_cred,
             SDATA (keyfile),
             file_format);

          if (ret < GNUTLS_E_SUCCESS)
            return gnutls_make_error (ret);
	}
    }

  GNUTLS_INITSTAGE (proc) = GNUTLS_STAGE_FILES;

  GNUTLS_LOG (1, max_log_level, "gnutls_init");

  ret = gnutls_init (&state, GNUTLS_CLIENT);

  if (ret < GNUTLS_E_SUCCESS)
    return gnutls_make_error (ret);

  XPROCESS (proc)->gnutls_state = state;

  GNUTLS_INITSTAGE (proc) = GNUTLS_STAGE_INIT;

  GNUTLS_LOG (1, max_log_level, "setting the priority string");

  ret = gnutls_priority_set_direct(state,
                                   (char*) SDATA (priority_string),
                                   NULL);

  if (ret < GNUTLS_E_SUCCESS)
    return gnutls_make_error (ret);

  GNUTLS_INITSTAGE (proc) = GNUTLS_STAGE_PRIORITY;

  if (EQ (type, Qgnutls_x509pki))
    {
      ret = gnutls_cred_set (state, GNUTLS_CRD_CERTIFICATE, x509_cred);
    }
  else if (EQ (type, Qgnutls_anon))
    {
      ret = gnutls_cred_set (state, GNUTLS_CRD_ANON, anon_cred);
    }
  else
    {
      error ("unknown credential type");
      ret = GNUTLS_EMACS_ERROR_INVALID_TYPE;
    }

  if (ret < GNUTLS_E_SUCCESS)
    return gnutls_make_error (ret);

  XPROCESS (proc)->gnutls_anon_cred = anon_cred;
  XPROCESS (proc)->gnutls_x509_cred = x509_cred;
  XPROCESS (proc)->gnutls_cred_type = type;

  GNUTLS_INITSTAGE (proc) = GNUTLS_STAGE_CRED_SET;

  emacs_gnutls_handshake (XPROCESS (proc));

  return gnutls_make_error (GNUTLS_E_SUCCESS);
}

DEFUN ("gnutls-bye", Fgnutls_bye,
       Sgnutls_bye, 2, 2, 0,
       doc: /* Terminate current GNU TLS connection for PROCESS.
The connection should have been initiated using `gnutls-handshake'.

If CONT is not nil the TLS connection gets terminated and further
receives and sends will be disallowed. If the return value is zero you
may continue using the connection.  If CONT is nil, GnuTLS actually
sends an alert containing a close request and waits for the peer to
reply with the same message.  In order to reuse the connection you
should wait for an EOF from the peer.

This function may also return `gnutls-e-again', or
`gnutls-e-interrupted'.  */)
    (Lisp_Object proc, Lisp_Object cont)
{
  gnutls_session_t state;
  int ret;

  CHECK_PROCESS (proc);

  state = XPROCESS (proc)->gnutls_state;

  ret = gnutls_bye (state,
                    NILP (cont) ? GNUTLS_SHUT_RDWR : GNUTLS_SHUT_WR);

  return gnutls_make_error (ret);
}

void
syms_of_gnutls (void)
{
  global_initialized = 0;

  Qgnutls_code = intern_c_string ("gnutls-code");
  staticpro (&Qgnutls_code);

  Qgnutls_anon = intern_c_string ("gnutls-anon");
  staticpro (&Qgnutls_anon);

  Qgnutls_x509pki = intern_c_string ("gnutls-x509pki");
  staticpro (&Qgnutls_x509pki);

  Qgnutls_e_interrupted = intern_c_string ("gnutls-e-interrupted");
  staticpro (&Qgnutls_e_interrupted);
  Fput (Qgnutls_e_interrupted, Qgnutls_code,
        make_number (GNUTLS_E_INTERRUPTED));

  Qgnutls_e_again = intern_c_string ("gnutls-e-again");
  staticpro (&Qgnutls_e_again);
  Fput (Qgnutls_e_again, Qgnutls_code,
        make_number (GNUTLS_E_AGAIN));

  Qgnutls_e_invalid_session = intern_c_string ("gnutls-e-invalid-session");
  staticpro (&Qgnutls_e_invalid_session);
  Fput (Qgnutls_e_invalid_session, Qgnutls_code,
        make_number (GNUTLS_E_INVALID_SESSION));

  Qgnutls_e_not_ready_for_handshake =
    intern_c_string ("gnutls-e-not-ready-for-handshake");
  staticpro (&Qgnutls_e_not_ready_for_handshake);
  Fput (Qgnutls_e_not_ready_for_handshake, Qgnutls_code,
        make_number (GNUTLS_E_APPLICATION_ERROR_MIN));

  defsubr (&Sgnutls_get_initstage);
  defsubr (&Sgnutls_errorp);
  defsubr (&Sgnutls_error_fatalp);
  defsubr (&Sgnutls_error_string);
  defsubr (&Sgnutls_boot);
  defsubr (&Sgnutls_deinit);
  defsubr (&Sgnutls_bye);
}
#endif
