/**
 * @file media-gst.h Media API
 * @ingroup core
 */

/* purple
 *
 * Purple is the legal property of its developers, whose names are too numerous
 * to list here.  Please refer to the COPYRIGHT file distributed with this
 * source distribution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __MEDIA_GST_H_
#define __MEDIA_GST_H_

#include "media.h"
#include "mediamanager.h"

#include <gst/gst.h>

G_BEGIN_DECLS

/** @copydoc _PurpleMediaElementInfo */
typedef struct _PurpleMediaElementInfo PurpleMediaElementInfo;

typedef enum {
	PURPLE_MEDIA_ELEMENT_AUDIO = 1,			/** supports audio */
	PURPLE_MEDIA_ELEMENT_VIDEO = 1 << 1,		/** supports video */
	PURPLE_MEDIA_ELEMENT_AUDIO_VIDEO = PURPLE_MEDIA_ELEMENT_AUDIO
			| PURPLE_MEDIA_ELEMENT_VIDEO, 	/** supports audio and video */

	PURPLE_MEDIA_ELEMENT_NO_SRCS = 0,		/** has no src pads */
	PURPLE_MEDIA_ELEMENT_ONE_SRC = 1 << 2,		/** has one src pad */
	PURPLE_MEDIA_ELEMENT_MULTI_SRC = 1 << 3, 	/** has multiple src pads */
	PURPLE_MEDIA_ELEMENT_REQUEST_SRC = 1 << 4, 	/** src pads must be requested */

	PURPLE_MEDIA_ELEMENT_NO_SINKS = 0,		/** has no sink pads */
	PURPLE_MEDIA_ELEMENT_ONE_SINK = 1 << 5, 	/** has one sink pad */
	PURPLE_MEDIA_ELEMENT_MULTI_SINK = 1 << 6, 	/** has multiple sink pads */
	PURPLE_MEDIA_ELEMENT_REQUEST_SINK = 1 << 7, 	/** sink pads must be requested */

	PURPLE_MEDIA_ELEMENT_UNIQUE = 1 << 8,		/** This element is unique and
							 only one instance of it should
							 be created at a time */

	PURPLE_MEDIA_ELEMENT_SRC = 1 << 9,		/** can be set as an active src */
	PURPLE_MEDIA_ELEMENT_SINK = 1 << 10,		/** can be set as an active sink */
} PurpleMediaElementType;

struct _PurpleMediaElementInfo
{
	const gchar *id;
	PurpleMediaElementType type;
	GstElement *(*create)(void);
};

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Gets the source from a session
 *
 * @param media The media object the session is in.
 * @param sess_id The session id of the session to get the source from.
 *
 * @return The source retrieved.
 */
GstElement *purple_media_get_src(PurpleMedia *media, const gchar *sess_id);

/**
 * Gets the pipeline from the media session.
 *
 * @param media The media session to retrieve the pipeline from.
 *
 * @return The pipeline retrieved.
 */
GstElement *purple_media_get_pipeline(PurpleMedia *media);

/**
 * Gets the tee from a given session/stream.
 *
 * @param media The instance to get the tee from.
 * @param session_id The id of the session to get the tee from.
 * @param participant Optionally, the participant of the stream to get the tee from.
 *
 * @return The GstTee element from the chosen session/stream.
 */
GstElement *purple_media_get_tee(PurpleMedia *media,
		const gchar *session_id, const gchar *participant);


/**
 * Gets the pipeline from the media manager.
 *
 * @param manager The media manager to get the pipeline from.
 *
 * @return The pipeline.
 */
GstElement *purple_media_manager_get_pipeline(PurpleMediaManager *manager);

/**
 * Returns a GStreamer source or sink for audio or video.
 *
 * @param manager The media manager to use to obtain the source/sink.
 * @param type The type of source/sink to get.
 */
GstElement *purple_media_manager_get_element(PurpleMediaManager *manager,
		PurpleMediaSessionType type);

PurpleMediaElementInfo *purple_media_manager_get_element_info(
		PurpleMediaManager *manager, const gchar *name);
gboolean purple_media_manager_register_element(PurpleMediaManager *manager,
		PurpleMediaElementInfo *info);
gboolean purple_media_manager_unregister_element(PurpleMediaManager *manager,
		const gchar *name);
gboolean purple_media_manager_set_active_element(PurpleMediaManager *manager,
		PurpleMediaElementInfo *info);
PurpleMediaElementInfo *purple_media_manager_get_active_element(
		PurpleMediaManager *manager, PurpleMediaElementType type);

#ifdef __cplusplus
}
#endif

G_END_DECLS

#endif  /* __MEDIA_GST_H_ */
